const express = require('express');
const cors = require('cors');
const helmet = require('helmet');
const morgan = require('morgan');
const http = require('http');
const cookieParser = require('cookie-parser');
const path = require('path');

// Load environment variables from the root directory
require('dotenv').config({ path: path.join(__dirname, '../.env') });

const authRoutes = require('./routes/authRoutes');
const eventRoutes = require('./routes/eventRoutes');
const userRoutes = require('./routes/userRoutes');
const forumRoutes = require('./routes/forumRoutes');
const exportRoutes = require('./routes/exportRoutes');
const errorHandler = require('./middleware/errorMiddleware');

const app = express();
const server = http.createServer(app);

// Trust proxy is required for secure cookies behind a reverse proxy (Heroku, Nginx, etc.)
app.set('trust proxy', 1);

// Build allowed CORS origins list (normalize trailing slash)
const buildAllowedOrigins = () => {
  const defaults = [
    'http://localhost:5173',
    'http://localhost:3001',
    'http://localhost:3000',
    "https://aeer-ua.com.ufm-madagascar.org",
    "https://site.aeer.ufm-madagascar.org",
    "https://aeer-ua.app",
    "https://www.aeer-ua.app"
  ];
  const env = process.env.FRONTEND_URL;
  const list = [...defaults];
  if (env && typeof env === 'string') {
    for (const raw of env.split(',')) {
      const base = raw.trim();
      if (!base) continue;
      const noSlash = base.replace(/\/$/, '');
      list.push(noSlash, `${noSlash}/`);
    }
  }
  // Common production domains (prevent mismatch due to slash)
  const prodCandidates = [
    'https://admin.interface.ufm-madagascar.org',
    'https://backend-true.ufm.ufm-madagascar.org',
    'https://aeer-ua.com.ufm-madagascar.org',
    'https://aeer.back.ufm-madagascar.org'
  ];
  for (const c of prodCandidates) {
    list.push(c, `${c}/`);
  }
  return Array.from(new Set(list));
};

const allowedOrigins = buildAllowedOrigins();

// CORS Middleware - Place before other middleware
app.use(cors({
  origin: (origin, callback) => {
    // Allow requests with no origin (like mobile apps or curl requests)
    if (!origin) return callback(null, true);
    
    // Check if origin is in the allowed list
    if (allowedOrigins.includes(origin)) {
      return callback(null, true);
    }

    // Allow any localhost origin for development
    if (origin.startsWith('http://localhost:')) {
      return callback(null, true);
    }

    // Allow specific production domains if not in list (safety net)
    if (origin.endsWith('.ufm-madagascar.org')) {
        return callback(null, true);
    }

    console.log('Blocked by CORS:', origin);
    callback(new Error('Not allowed by CORS'));
  },
  credentials: true
}));

// Middleware
app.use(helmet({
  contentSecurityPolicy: {
    useDefaults: false,
    directives: {
      defaultSrc: ["'self'", "https://site.aeer.ufm-madagascar.org", "https://aeer-ua.app"],
      connectSrc: ["'self'", "https://api.aeer.ufm-madagascar.org", "https://site.aeer.ufm-madagascar.org", "https://aeer-ua.app", "http://localhost:3000", "http://localhost:5173"],
      imgSrc: ["'self'", "data:", "https://picsum.photos", "https://images.unsplash.com", "https://site.aeer.ufm-madagascar.org", "https://aeer-ua.app"],
      scriptSrc: ["'self'", "'unsafe-inline'", "'unsafe-eval'", "https://site.aeer.ufm-madagascar.org", "https://aeer-ua.app"],
      styleSrc: ["'self'", "'unsafe-inline'", "https://site.aeer.ufm-madagascar.org", "https://aeer-ua.app"],
      fontSrc: ["'self'", "data:", "https://fonts.gstatic.com"],
      objectSrc: ["'none'"],
      upgradeInsecureRequests: null,
    },
  },
}));
app.use(express.json());
app.use(cookieParser());
app.use(morgan('dev'));

// Routes
app.get('/api/health', (req, res) => res.json({ status: 'ok', timestamp: new Date().toISOString() }));
app.use('/api/auth', authRoutes);
app.use('/api/events', eventRoutes);
app.use('/api/users', userRoutes);
app.use('/api/forum', forumRoutes);
app.use('/api/export', exportRoutes);

// API 404 Handler - Catch any unmatched API requests
app.use('/api/*', (req, res) => {
  res.status(404).json({ 
      message: 'API endpoint not found', 
      path: req.originalUrl,
      method: req.method 
  });
});

// Serve static files from the React app
app.use(express.static(path.join(__dirname, '../public')));

// Handle React routing, return all requests to React app
app.get('*', (req, res, next) => {
  // Skip API routes so they can return 404 if not found
  if (req.path.startsWith('/api')) {
    return next();
  }

  // Debug: Log if a non-API route is requested with JSON accept header
  if (req.headers.accept && req.headers.accept.includes('application/json')) {
      console.warn(`[Warning] Client requested JSON for non-API route: ${req.path}. Possible proxy misconfiguration (stripping /api prefix?)`);
  }

  res.sendFile(path.join(__dirname, '../public/index.html'));
});

// Error Handling
app.use(errorHandler);

const PORT = process.env.PORT || 3000;

server.listen(PORT, () => {
  console.log(`Server running on port ${PORT}`);
});

module.exports = { app };
