const bcrypt = require('bcryptjs');
const jwt = require('jsonwebtoken');
const crypto = require('crypto');
const db = require('../config/db');

const register = async (req, res, next) => {
  const { username, email, password, role } = req.body;

  try {
    // Check if user exists
    const userCheck = await db.query('SELECT * FROM users WHERE email = $1', [email]);
    if (userCheck.rows.length > 0) {
      return res.status(400).json({ message: 'User already exists' });
    }

    // Hash password
    const salt = await bcrypt.genSalt(10);
    const hashedPassword = await bcrypt.hash(password, salt);

    // Get role ID (default to participant if not specified or invalid, but for now let's require valid role or default)
    let roleName = role || 'participant';
    const roleResult = await db.query('SELECT id FROM roles WHERE name = $1', [roleName]);
    
    if (roleResult.rows.length === 0) {
        return res.status(400).json({ message: 'Invalid role' });
    }
    const roleId = roleResult.rows[0].id;

    // Insert user
    const newUser = await db.query(
      'INSERT INTO users (username, email, password_hash, role_id) VALUES ($1, $2, $3, $4) RETURNING id, username, email',
      [username, email, hashedPassword, roleId]
    );

    res.status(201).json({ message: 'User registered successfully', user: newUser.rows[0] });
  } catch (error) {
    next(error);
  }
};

const login = async (req, res, next) => {
  const { email, password } = req.body;
  console.log('Login attempt:', { email, passwordProvided: !!password });

  try {
    const result = await db.query(`
      SELECT u.id, u.username, u.email, u.password_hash, r.name as role_name 
      FROM users u 
      JOIN roles r ON u.role_id = r.id 
      WHERE u.email = $1
    `, [email]);

    if (result.rows.length === 0) {
      console.log('Login failed: User not found');
      return res.status(400).json({ message: 'Email ou mot de passe incorrect' });
    }

    const user = result.rows[0];
    const isMatch = await bcrypt.compare(password, user.password_hash);

    if (!isMatch) {
      console.log('Login failed: Password mismatch');
      return res.status(400).json({ message: 'Email ou mot de passe incorrect' });
    }

    const token = jwt.sign(
      { id: user.id, username: user.username, role: user.role_name },
      process.env.JWT_SECRET,
      { expiresIn: '1h' }
    );

    // Determine environment based on origin or NODE_ENV
    // If origin is not localhost, assume production (HTTPS) for cross-site cookies
    const origin = req.headers.origin || '';
    const isLocalhost = origin.includes('localhost') || origin.includes('127.0.0.1');
    const isProduction = process.env.NODE_ENV === 'production' || !isLocalhost;

    // Set cookie
    res.cookie('token', token, {
      httpOnly: true,
      secure: isProduction, // Use secure in production or non-localhost
      sameSite: isProduction ? 'none' : 'lax',
      maxAge: 3600000, // 1 hour
      path: '/'
    });

    res.json({ message: 'Login successful', user: { id: user.id, username: user.username, email: user.email, role: user.role_name } });
  } catch (error) {
    next(error);
  }
};

const getMe = async (req, res, next) => {
  try {
    // Prevent caching of the user data
    res.set('Cache-Control', 'no-store, no-cache, must-revalidate, proxy-revalidate');
    res.set('Pragma', 'no-cache');
    res.set('Expires', '0');

    // User is already attached to req by middleware
    const user = req.user;
    res.json({ user });
  } catch (error) {
    next(error);
  }
};

const logout = (req, res) => {
  const origin = req.headers.origin || '';
  const isLocalhost = origin.includes('localhost') || origin.includes('127.0.0.1');
  const isProduction = process.env.NODE_ENV === 'production' || !isLocalhost;

  const cookieOptions = {
    httpOnly: true,
    secure: isProduction,
    sameSite: isProduction ? 'none' : 'lax',
    path: '/',
    expires: new Date(0) // Explicitly expire immediately
  };

  // Clear cookie with the same attributes as login
  res.cookie('token', '', cookieOptions);
  
  // Also try clearing without SameSite/Secure just in case of mismatch
  res.clearCookie('token', { path: '/' });

  res.json({ message: 'Logged out successfully' });
};

const getAllUsers = async (req, res, next) => {
  try {
    const result = await db.query(`
      SELECT u.id, u.username, u.email, u.created_at, r.name as role_name 
      FROM users u 
      JOIN roles r ON u.role_id = r.id
      ORDER BY u.created_at DESC
    `);
    res.json(result.rows);
  } catch (error) {
    next(error);
  }
};

const changePassword = async (req, res, next) => {
  const { currentPassword, newPassword } = req.body;
  const userId = req.user.id;

  try {
    // Get user
    const result = await db.query('SELECT * FROM users WHERE id = $1', [userId]);
    if (result.rows.length === 0) {
      return res.status(404).json({ message: 'Utilisateur non trouvé' });
    }
    const user = result.rows[0];

    // Verify current password
    const isMatch = await bcrypt.compare(currentPassword, user.password_hash);
    if (!isMatch) {
      return res.status(400).json({ message: 'Mot de passe actuel incorrect' });
    }

    // Hash new password
    const salt = await bcrypt.genSalt(10);
    const hashedPassword = await bcrypt.hash(newPassword, salt);

    // Update password
    await db.query('UPDATE users SET password_hash = $1 WHERE id = $2', [hashedPassword, userId]);

    res.json({ message: 'Mot de passe modifié avec succès' });
  } catch (error) {
    next(error);
  }
};

const forgotPassword = async (req, res, next) => {
  const { email } = req.body;

  try {
    const result = await db.query('SELECT * FROM users WHERE email = $1', [email]);
    if (result.rows.length === 0) {
      // For security, don't reveal if user exists, but for UX we might want to say "If email exists..."
      // Here we will just say sent.
      return res.json({ message: 'Si cet email existe, un lien de réinitialisation a été envoyé.' });
    }

    const resetToken = crypto.randomBytes(32).toString('hex');
    const resetTokenExpiry = new Date(Date.now() + 3600000); // 1 hour

    await db.query('UPDATE users SET reset_token = $1, reset_token_expiry = $2 WHERE email = $3', [resetToken, resetTokenExpiry, email]);

    // Mock Email Sending
    console.log(`[MOCK EMAIL] Password reset link for ${email}: http://localhost:5173/reset-password?token=${resetToken}`);

    res.json({ message: 'Si cet email existe, un lien de réinitialisation a été envoyé.' });
  } catch (error) {
    next(error);
  }
};

const resetPassword = async (req, res, next) => {
  const { token, newPassword } = req.body;

  try {
    const result = await db.query('SELECT * FROM users WHERE reset_token = $1 AND reset_token_expiry > NOW()', [token]);
    
    if (result.rows.length === 0) {
      return res.status(400).json({ message: 'Lien invalide ou expiré' });
    }

    const user = result.rows[0];

    // Hash new password
    const salt = await bcrypt.genSalt(10);
    const hashedPassword = await bcrypt.hash(newPassword, salt);

    // Update password and clear token
    await db.query('UPDATE users SET password_hash = $1, reset_token = NULL, reset_token_expiry = NULL WHERE id = $2', [hashedPassword, user.id]);

    res.json({ message: 'Mot de passe réinitialisé avec succès' });
  } catch (error) {
    next(error);
  }
};

module.exports = { register, login, getMe, logout, getAllUsers, changePassword, forgotPassword, resetPassword };
